/* sched.c:  Creates PostScript program to print class schedules
 * sched.pro:  prologue of PostScript routines.
 * 
 *  Usage:  sched < sched.dat | lpr
 *
 *  Format of sched.dat file:
 *
 *     First Line:   Title1|Title2
 *     Subsequent:   Title|Professor|Type|Location|days|time|length|priority
 *
 *     Where:  Title is title of class. (eg.  English 1A.19)
 *             Professor is a name. (eg. Hilfinger)
 *             Type is type of class.  (eg.  Lecture, Lab, or Discussion)
 *             Location is room.  (eg.  24 Wheeler)
 *             days is days.  (eg. m, tu, w, th, f)
 *             time is starting time (eg.  8 thru 5)
 *             length is length of class in hours (eg. 1.5)
 *             Priority is 1 or 2 (determines shade.  1 is higher)
 *
 *     For Example:
 *
 *         Alan Nishioka's|Fall 1987 Schedule
 *         Physics 7B|Sachs|Lec 2|1 PSL|mwf|11|1|1
 *         Physics 7B|Uwe Albertin|Lab 202|270 LeConte|w|8|3|2
 *         Physics 7B|Scott Cully|Dis 215|212 LeConte|f|10|1|2
 *         Math 50A|Sarason|Lec 1|100 Lewis|mwf|1|1|1
 *         Math 50A|Mark Davis|Dis 101|3 Evans|tuth|8|1|2
 *         EECS 40|John Gamelin|Dis 102|2111 Etcheverry|m|3|1|2
 *         EECS 40|Cheung|Lec 1|102 Wurster|tuth|9.5|1.5|1
 *         EECS 43|Michelle Wong|Lab 16|143 Cory|w|2|2|2
 *         EECS 90|Birdsall|Sem 1|4 LeConte|m|4|1|1
 *         CS 60C|Hilfinger/Clancy|Lec 1|1 PSL|tuth|4|1.5|1
 *         CS 60C|Dave Parker|Dis 107|3109 Etch|th|3|1|2
 *         CS 60C|Pete Klier|Dis 101|5 Evans|w|4|1|2
 *
 *     Alan Nishioka      ch125-dv@violet.berkeley.edu
 *     974 Tulare Avenue
 *     Albany, CA 94707
 */

#include<stdio.h>
#include<strings.h>
#define MAXWORDLEN 50

char *progname;

main( argc, argv )
     int argc;
     char **argv;
{
    void fmt_error(), skip_term();
    FILE *fp, *fopen();
    char header1[ MAXWORDLEN ], header2[ MAXWORDLEN ];
    char class[ MAXWORDLEN ], professor[ MAXWORDLEN ], section[ MAXWORDLEN ];
    char room[ MAXWORDLEN ], days[ MAXWORDLEN ], *day;
    int priority = 0, i, c, line = 0, status;
    float length = 0, time = 0;

    progname = argv[0];
    if((fp = fopen( "sched.pro", "r")) == NULL) {
	perror( argv[0] );
	exit( 1);
    }
    while((c = getc( fp)) != EOF) putchar( c);
    fclose( fp);
    
    ++line;
    if( scan_string( stdin, header1 ) != 1 ) fmt_error( line );
    if( scan_string( stdin, header2 ) != 1 ) fmt_error( line );
    printf( "(%s) (%s) draw_header\n", header1, header2 );
    
    while( 1 ) {
	++line;
	if(( status = scan_string( stdin, class )) == EOF ) break;
	else if( status != 1 ) fmt_error( line );
	if( scan_string( stdin, professor ) != 1 ) fmt_error( line );
	if( scan_string( stdin, section ) != 1 ) fmt_error( line );
	if( scan_string( stdin, room ) != 1 ) fmt_error( line );
	if( scan_string( stdin, days ) != 1 ) fmt_error( line );
	if( fscanf( stdin, "%f|%f|%d", &time, &length, &priority ) != 3 )
	    fmt_error( line );
	skip_term( fp );
	for( i = 0; i < strlen( days ); i++ ) {
	    switch( days[i] ) {
	    case 'm':
		day = "mon";
		break;
	    case 'w':
		day = "wed";
		break;
	    case 'f':
		day = "fri";
		break;
	    case 't':
		if( days[i+1] == 'u') {
		    day = "tue";
		    i++;
		    break;
		}
		else if( days[i+1] == 'h') {
		    day = "thu";
		    i++;
		    break;
		}
	    default:
		fmt_error( line );
	    }
	    printf( "(%s) (%s) (%s) (%s) ", class, professor, section, room );
	    printf( "%d %1.1f hour %s %1.1f entry\n",
		   priority, length, day, time );
	}
    }
    printf( "showpage\n" );
}

/* void skip_term( FILE *fp ) --
 *     skips of terminators: '|', ' ', '\t', '\n'
 * Return: number of chars skipped or EOF
 */
void skip_term( fp )
     FILE *fp;
{
    char c;
    
    while( 1 ) {
	c = getc( fp );
	if( c ==  ' ' ) continue;
	else if( c == '\n' ) continue;
	else if( c == '\t' ) continue;
	else if( c == '|' ) continue;
	else if( c == EOF ) return;
	else {
	    ungetc( c, fp );
	    return;
	}
    }
}

/* int scan_string( FILE *fp, char *string ) --
 *     skips intervening terminators,
 * scans a string terminated by '|' '\t' or '\n' (not space),
 * skips subsequent terminators.
 * Returns: 1 if string scanned, EOF if End of File.
 */
int scan_string( fp, string ) 
     FILE *fp;
     char *string;
{
    int status;
    
    skip_term( fp );
    status = fscanf( fp, "%[^|\t\n]", string );
    skip_term( fp );
    return( status );
}

void fmt_error( line )
     int line;
{
    fprintf(stderr, "%s:format error on line %d\n", progname, line );
    exit(1);
}
