#include <stdio.h>

#define WIDTH 240		/* Width and Height of LCD screen */
#define PADW 16			/* Width of padding on a line */
#define HEIGHT 64
#define MAXLEN 256		/* Maximum string length */
#define SH(a) (((a)+15)>>4)		/* Number of shorts to hold "a" bits */
#define CH(a) (((a)+ 7)>>3)		/* Number of  chars to hold "a" bits */
#define IBM

#ifdef IBM

#define COM1 0x3f8
#define COM2 0x2f8
#define COM3 0x3e8
#define COM4 0x2e8
#define SERIAL COM2
#include <bios.h>
#include <malloc.h>
#define READ "rb"
short huge *bits;
#define SEND(x,y)	ibmsend((x))

#else

#define READ "r"
#define WRITE "w"
#define SERIAL "SER:"
#define SEEK_SET 0
char *malloc();
short *bits;
#define SEND(x,y)	putc((x),(y))

#endif

long bits_size;

int debug = 0;			/* debug flag */

main( argc, argv )
    int argc;
    char **argv;
{
    FILE *fp, *fd;
    int i, j, bitw, bith, yoff = 0, status, invert = 0;
    char input[ MAXLEN ];

    if( argc == 1 || argc > 4 ) {
        printf( "Usage: lcd3 filename [y_offset] [invert]\n" );
        exit( 1 );
    }

    if( argc > 2 ) yoff = atoi( argv[ 2 ] );
    if( argc > 3 ) invert = atoi( argv[ 3 ] );

    fd = fopen( argv[ 1 ], READ );
    if( fd == 0 ) {
       printf( "Can't open file %s.\n", argv[ 1 ] );
       exit( 1 );
    }

    status = OpenXBM( fd, &bitw, &bith );
    if( status == 1 ) status = OpenXWD( fd, &bitw, &bith );
    if( status == 1 ) status = OpenPCX( fd, &bitw, &bith );
    if( status == 1 ) status = OpenMAC( fd, &bitw, &bith );
    if( status == 1 ) exit( 1 );
    else if( status == EOF ) printf( "Premature EOF.\n" );

#ifdef IBM
    outp( SERIAL+3, 0x80 );			/* Set DLAB */
	outp( SERIAL, 6 );				/* 19.2k baud */
	outp( SERIAL+1, 0 );
    outp( SERIAL+3, 0x03 );			/* no parity, 8 bits, 1 stop bit */
#else
    fp = fopen( SERIAL, WRITE );
    if( fp == 0 ) {
        printf( "Can't open serial for output.\n" );
        exit( 1 );
    }
#endif

    for(;;) {
        printf( "Please enter a Y offset.  Enter q to quit. d, b, k, i\n" );
        fgets( input, MAXLEN-1, stdin );
        if( input[ 0 ] == 'q' ) break;
		else if( input[ 0 ] == 'd' ) {
		    for( i = 0; i < bits_size; i += 1 ) {
			    printf( "%.4hX", bits[ i ] );
				if(( i & 15 ) == 15 ) printf( "\n" );
				else printf( " " );
			}
		    continue;
		}
		else if( input[ 0 ] == 'b' ) {
		    bars( 0, fp );
			continue;
		}
		else if( input[ 0 ] == 'k' ) {
		    bars( 1, fp );
			continue;
		}
		else if( input[ 0 ] == 'i' ) {
			invert = !invert;
			printf( "invert = %d\n", invert );
			continue;
		}

        yoff = atoi( input );

        printf( "Downloading bitmap to LCD screen...\n" );

        SEND( 0x30, fp );		/* Clear address counter */
    					        /* Set latch to load mode */
 
        for( j = 1; j < HEIGHT; j += 1 ) {	/* All but the first line */
	        output_line( j, bitw, yoff, invert, fp );
	        for( i = 0; i < PADW - 1; i += 1 ) SEND( 0x20, fp ); /* Post pad */
        }
        output_line( 0, bitw, yoff, invert, fp ); /* Put first line last */
        SEND( 0, fp );			/* Clear latch to display mode */
		fflush( fp );
        printf( "Done.\n" );
    }
#ifdef IBM
    if( bits != 0 ) hfree( bits );
#else
    if( bits != 0 ) free( (char *)bits );
#endif
    fclose( fp );
    fclose( fd );
}

output_line( j, bitw, yoff, invert, fp )
    int j, bitw, yoff, invert;
    FILE *fp;
{
    int i, k, shortw, c;
    short bitul, bitur, bitll, bitlr;

    shortw = SH( bitw );		/* Width of bitmap in shorts */

	SEND( 0x20, fp );			/* One bit pre pad */

    if( debug < 0 ) printf( "%d:\n", j );
    for( i = 0; i < SH( WIDTH ); i += 1 ) {
        bitul = bits[ ( j + yoff ) * shortw + i ];
	    bitur = bits[ ( j + yoff ) * shortw + SH( WIDTH ) + i ];
	    bitll = bits[ ( j + yoff + HEIGHT ) * shortw + i ];
	    bitlr = bits[ ( j + yoff + HEIGHT ) * shortw + SH( WIDTH ) + i ];

if( debug < 0 ) {
printf( "   %3x:%04x %3x:%04x %3x:%04x %3x:%04x\n",
        j * shortw + i, bitul, j * shortw + WIDTH / 16 + i, bitur,
	    ( j + HEIGHT ) * shortw + i, bitll,
	    ( j + HEIGHT ) * shortw + WIDTH / 16 + i, bitlr );
}

        for( k = 0; k < 16; k += 1 ) {
	        c = 0;
	        if(( i * 16 ) + k + 1 <= bitw || debug ) {
		        c |= bitul & 1; bitul >>= 1; c <<= 1;
		        c |= bitll & 1; bitll >>= 1; c <<= 1;
	        }
	        if(( i * 16 ) + k + WIDTH + 1 <= bitw || debug ) {
		        c |= bitur & 1; bitur >>= 1; c <<= 1;
	        	c |= bitlr & 1; bitlr >>= 1;
	        }
	        else c <<= 1;
	        if( !invert ) c ^= 0x0f;
	        SEND((char)( c | 0x20 ), fp );
        }
    }
}

long int getint( fp )
    /* Gets a long int from fp MSB first */
    FILE *fp;
{
    int c, i;
    long int value = 0;

    for( i = 0; i < 4; i +=1 ) {
		value <<= 8;
        c = getc( fp );
        if( c == EOF ) return( EOF );
		value |= c & 0xff;
    }
    return( value );
}

short getshort( fp )
    /* Gets a short int from fp MSB first */
    FILE *fp;
{
    int c, i;
    short value = 0;

    for( i = 0; i < 2; i +=1 ) {
		value <<= 8;
        c = getc( fp );
        if( c == EOF ) return( EOF );
		value |= ( c & 0xff );
    }
    return( value );
}

getstring( name, fd )
    char *name;
    FILE *fd;
{
    int i;
 
    for( i = 0; i < MAXLEN; i += 1 ) {
        name[ i ] = getc( fd );
        if( name[ i ] == 0 ) return;
    }
}

OpenPCX( fd, bitw, bith )
    /* Read parameters from a PCX file */
	FILE *fd;
	int *bitw, *bith;
{
    int c, i, j, k, f, x, rl, b;
	unsigned short s;

    printf( "Reading PCX format..." );
	fseek( fd, 0L, SEEK_SET );					/* Find the start of file */

    if( getc( fd ) != 0x0a || getc( fd ) == EOF || getc( fd ) != 0x01 ||
	    getc( fd ) != 0x01 ) {
        printf( "Error: Can't recognize this file type.\n", c );
        return( 1 );
    }
	for( i = 0; i < 4; i += 1 ) c = getc( fd );	/* Skip 5 chars */
	if( c == EOF ) return( EOF );

	c = getc( fd );
    *bitw = ( getc( fd ) << 8 ) + c + 1;
    printf( " width = %hd,", *bitw );

	c = getc( fd );
    *bith = ( getc( fd ) << 8 ) + c;
    printf( " height = %hd\n", *bith );
	for( i = 0; i < 116; i += 1 ) c = getc( fd ); /* Skip 116 chars */
	if( c == EOF ) return( EOF );

	bits_size = SH(*bitw) * (*bith);
#ifdef IBM
    bits = halloc( bits_size, sizeof( short));		/* Allocate space */
#else
    bits = (short *)malloc( (unsigned)(bits_size * sizeof( short )) );
#endif
    if( bits == 0 ) {
        printf( "Can't allocate %ld shorts bits space.\n", bits_size );
        return( 1 );
    }
	else printf( "Allocated %ld shorts.\n", bits_size );

    for( j = 0; j < (*bith); j += 1 ) {			/* For each row */
        for( i = 0, f = 0, x = 0; x < (*bitw);) { 
		    c = getc( fd );
/*printf( "start: c = %.2x, i = %d, j = %d\n", c, i, j );*/
		    if( c == EOF ) return( EOF );
		    if(( c & 0xc0 ) == 0xc0 ) {		/* If a Run length */
			    rl = ( c & 0x3f );
			    c = getc( fd );
/*printf( "middle: %.2x\n", c );*/
			}
		    else rl = 1;
			s = 0;
			for( b = 0; b < 8; b += 1 ) {
			    s <<= 1;
			    s |= ( c & 1 );
				c >>= 1;
			}

		    for( k = 0; k < rl; k += 1 ) {
		    if( f == 0 ) bits[ i + (j * SH(*bitw)) ] = s;
				else {
			    	bits[ i + (j * SH(*bitw)) ] |= ( s << 8 );
/*printf( "bits[ %d ] = %.4hx\n", i+(j*SH(*bitw)), bits[ i + (j * SH(*bitw)) ] );*/
			        i += 1;
				}
/*printf( "x = %d, i = %d, j = %d, c = %.2x\n", x, i, j, c );*/
		        f = 1 - f;
				x += 8;
		   }
	    }
    }
    return( 0 );
}

OpenXWD( fd, bitw, bith )
    /* Read parameters from an XWD file */
    FILE *fd;
    int *bitw, *bith;
{
    char name[ MAXLEN ];
    long int value;
    int c, i, space;

    printf( "Reading XWD format..." );
    getint( fd );			/* header size */
    value = getint( fd );		/* file version */
    if( value != 6 ) {
        printf( "Error: Can't handle a type %ld XWD file.\n", value );
        return( 1 );
    }
    getint( fd );			/* display type */
    value = getint( fd );		/* display planes */
    if( value != 1 ) {
        printf( "Error: Too many planes in xwd file.\n" );
        return( 1 );
    }
    getint( fd );			/* pixmap format */
    getint( fd );			/* pixmap width */
    getint( fd );			/* pixmap heigth */
    printf( " width = %hd,", *bitw = getshort( fd ));
    printf( " height = %hd,", *bith = getshort( fd ));
    getshort( fd );			/* window x */
    getshort( fd );			/* window y */
    getshort( fd );			/* window border width */
    getshort( fd );			/* window ncolors */
    getstring( name, fd );
    printf( " title: %s\n", name );

    if( debug ) for( i = 0; i < debug; i += 1 ) getc( fd );
	bits_size = SH( *bitw ) * (*bith);
#ifdef IBM
    	bits = halloc( bits_size, sizeof( short));	/* Allocate space */
#else
    	bits = (short *)malloc( (unsigned)(bits_size * sizeof( short )) );
#endif
    if( bits == 0 ) {
        printf( "Can't allocate %ld shorts bits space.\n", bits_size );
        return( 1 );
    }
    else printf( "Allocated %ld shorts.\n", bits_size );
    space = 0;
    for( i = 0; i < (( *bitw + 15 ) / 16 ) * (*bith); i += 1 ){	/* Fill it */
        bits[ i ] = getshort( fd );
        if( debug ) {
            if( space == 0 ) printf( "%04x: ", i );
            printf( "%04x", bits[ i ]);
            space += 1;
            if( space % 2 == 0 ) printf( " " );
            if( space >= 8 ) {
                 printf( "\n" );
                 space = 0;
            } 
        }
    }
    if( debug ) printf( "At End, i = %d, bits[ i ] = %d\n", i, bits[ i ] );
    return( 0 );
}

skip_to_x( fp )
    FILE *fp;
{
	int c;

    for(;;) {
		c = getc( fp );
	    if( c == 'x' || c == EOF ) return( c );
	}
}

OpenXBM( fd, bitw, bith )
    /* Read parameters from an X bitmap file */
    FILE *fd;
    int *bitw, *bith;
{
    char type[ MAXLEN ], word[ MAXLEN ];
    long int value;
    int c, i, j, lo;
    short data;

    c = getc( fd );
    if( c != '#' ) {
         printf( "Not X bitmap format... Trying XWD format.\n" );
         ungetc( c, fd );
         return( 1 );
    }
    printf( "Reading X bitmap format..." );
    fscanf( fd, "%*s%*s%d%*s%*s%d", bitw, bith );  /* Read header information */
    fscanf( fd, "%s%s%*s%*s%*s", word, type );
    if( strcmp( word, "#define" ) == 0 ) {	   /* X10 format */
        fscanf( fd, "%*s%*s%s%*s%*s%*s", type );
    }
    printf( "width = %d, height = %d, type = %s\n", *bitw, *bith, type );

    bits_size = SH(*bitw) * (*bith);

#ifdef IBM
    bits = halloc( bits_size, sizeof( short ));
#else
    bits = (short *)malloc( (unsigned)( bits_size * sizeof( short ) ) );
#endif
							/* Allocate space */
    if( bits == 0 ) {
        printf( "Can't allocate %ld shorts bits space.\n", bits_size );
        return( 1 );
    }
    else printf( "Allocated %ld shorts.\n", bits_size );

    if( strcmp( "short", type ) == 0 ) {		/* X10 xbm format */
        for( i = 0; i < SH(*bitw) * (*bith); i += 1 ) {
			skip_to_x( fd );
            fscanf( fd, "%x", &data );
            bits[ i ] = data;
        }
        if( debug )
            printf( "At End, i = %d, bits[ i ] = %d\n", i, bits[ i ] );
    }
    else if( strcmp( "char", type ) == 0 ) {	/* X11 xbm format */
        for( i = 0, j = 0; i < CH( *bitw ) * (*bith); i += 1 ) {
            if( fscanf( fd, "%x,", &lo ) != 1 ) {
                printf( "Error: Missing byte.\n" );
                return( EOF );
            }
            if( i & 1 ) {					/* Odd i */
                bits[ j ] |= lo << 8;
                bits[ j ] ^= 0xffff;	/* Invert bits */
                j += 1;
                if( debug ) printf( "%4d:%04x", j, bits[ j ] );
            }
            else {						/* Even i */
                bits[ j ] = lo;
                if( ( i + 1 ) % CH( *bitw ) == 0 ) j += 1;
            }
        }
    }
    else {
        printf( "Error: Wrong data type: %s\n", type );
        return( EOF );
    }
    return( 0 );
}

#ifdef IBM
ibmsend( c )
    int c;
{
	while(( inp( SERIAL+5 ) & 0x20 ) == 0 );		/* Busy Wait */

	outp( SERIAL, c );
}
#endif

bars( c, fp )
	int c;
    FILE *fp;
{
	int j, i;

    SEND( 0x30, fp );		/* Clear address counter */
    		 				/* Set latch to load mode */
 
    for( j = 0; j < HEIGHT; j += 1 ) {
	    SEND( 0x20, fp );			/* One bit pre pad */
	    for( i = 0; i < WIDTH; i += 1 ) 
		    if( c == 0 ) SEND( 0x20 | ((i&0x0f)^0x00), fp );
		    else if( c == 1 ) SEND( 0x20 | 0x0f, fp );
	    for( i = 0; i < PADW - 1; i += 1 ) SEND( 0x20, fp ); /* Post pad */
    }
    SEND( 0, fp );			/* Clear latch to display mode */
}

OpenMAC( fd, bitw, bith )
    /* Read parameters from a MacPaint file */
	FILE *fd;
	int *bitw, *bith;
{
    int        i, j, k, f, x, rl, b;
	unsigned short    s, swap;
	int     c, byte;

    printf( "Reading MacPaint format..." );
	fseek( fd, 0L, SEEK_SET );					/* Find the start of file */

    if( getc( fd ) != 0x00 ) {
        printf( "Error: Not MacPaint format.\n" );
        return( 1 );
    }
	for( i = 0; i < 512+128-1; i += 1 ) c = getc( fd );	/* Skip 639 chars */
	if( c == EOF ) return( EOF );

    *bitw = 576;
    *bith = 720;
	bits_size = SH(*bitw) * ((*bith) + 2);
#ifdef IBM
    bits = halloc( bits_size, sizeof( short));		/* Allocate space */
#else
    bits = (short *)malloc( (unsigned)(bits_size*2) );
#endif
    if( bits == 0 ) {
        printf( "Can't allocate %ld shorts bits space.\n", bits_size );
        return( 1 );
    }
	else printf( "Allocated %ld shorts.\n", bits_size );

    for( j = 0; j < (*bith); j += 1 ) {         /* For each row */
        for( i = 0, f = 0, x = 0; x < (*bitw);) { 
        	c = getc( fd );
        	if( c & 0x80 ) {             /* If a Run length */
                rl = 1;
                s = (-c) & 0xff;		/* negative of byte sized word */
                byte = getc( fd );
            }
            else {
                s = c;
                rl = 0;                /* RunLength is false */
            }

            for( k = 0; k < s + 1; k += 1 ) {
                if( !rl ) c = getc( fd );    /* Read a data byte */
                else c = byte;
                for( b = 0; b < 8; b += 1 ) {
                    bits[ i + (j * SH(*bitw)) ] <<= 1;
                    bits[ i + (j * SH(*bitw)) ] |= ( c & 1 );
                    c >>= 1;
                }  
                if( f ) {
                    swap = bits[ i + (j * SH(*bitw)) ];   /* Swap bytes */
                    bits[ i + (j * SH(*bitw)) ] = ( swap << 8 );
                    bits[ i + (j * SH(*bitw)) ] |= ( swap >> 8 );
                    bits[ i + (j * SH(*bitw)) ] ^= 0xffff;	/* Invert */
                    i += 1;
                }
                f = !f;
                x += 8;
            }
        }
    }
    return( 0 );
}
